/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.validation;

import javax.xml.stream.Location;
import javax.xml.stream.XMLStreamException;

import java.util.List;

import org.codehaus.stax2.validation.XMLValidationException;

import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorType;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeHolder;
import eu.europa.ec.taxud.cesop.readers.CesopParsingException;
import eu.europa.ec.taxud.cesop.readers.UnknownXsdVersionException;
import eu.europa.ec.taxud.cesop.readers.XmlLocation;

import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.createCustomError;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.createValidationError;

/**
 * Util class handling exceptions.
 */
public final class ExceptionHandler {
    private ExceptionHandler() {
    }

    /**
     * Handle exception validation error.
     *
     * @param e            the exception
     * @param messageRefId the message ref id
     * @return the validation error as defined by CESOP
     */
    public static ValidationError handleException(final Exception e, final String messageRefId) {
        final ValidationError validationError;
        if (e instanceof CesopValidationException) {
            validationError = ((CesopValidationException) e).getValidationError();
        } else if (e instanceof XMLValidationException) {
            validationError = handleXmlValidationException((XMLValidationException) e, messageRefId);
        } else if (e.getCause() instanceof XMLValidationException) {
            validationError = handleXmlValidationException((XMLValidationException) e.getCause(), messageRefId);
        } else if (e instanceof UnknownXsdVersionException || e instanceof CesopParsingException || e instanceof XMLStreamException) {
            validationError = createValidationError(ValidationErrorTypeEnum.CM_TR_0010, messageRefId);
            validationError.setErrorLongDesc(e.getMessage());
        } else {
            validationError = handleUnexpectedException(e, messageRefId);
        }
        return validationError;
    }

    private static ValidationError handleXmlValidationException(final XMLValidationException e, final String messageRefId) {
        final Location location = e.getValidationProblem().getLocation();
        ValidationErrorTypeEnum errorCode = deriveErrorCodeFromLocation(location);

        final ValidationErrorType validationError = ValidationErrorTypeHolder.INSTANCE.findByCode(errorCode.getCode());
        final String longDescription = validationError.getLongDescription()
                + "\nError in line: " + location.getLineNumber()
                + ", column" + ": " + location.getColumnNumber()
                + ": " + e.getValidationProblem().getMessage();
        final ValidationError result = createValidationError(errorCode, messageRefId);
        result.setErrorLongDesc(longDescription);
        return result;
    }

    private static ValidationErrorTypeEnum deriveErrorCodeFromLocation(Location location) {
        if (location instanceof XmlLocation) {
            final List<String> path = ((XmlLocation) location).getXmlPath();
            if (!path.isEmpty()) {
                String topElem = path.get(path.size() - 1);
                switch (topElem) {
                    case "DocRefId":
                        return ValidationErrorTypeEnum.CM_BR_0030;
                    case "CorrDocRefId":
                        return ValidationErrorTypeEnum.CM_BR_0040;
                    case "MessageRefId":
                        return ValidationErrorTypeEnum.MH_BR_0050;
                    case "CorrMessageRefId":
                        return ValidationErrorTypeEnum.MH_BR_0060;
                }
            }
        }
        return ValidationErrorTypeEnum.CM_TR_0010;
    }

    private static ValidationError handleUnexpectedException(final Exception e, final String messageRefId) {
        return createCustomError(messageRefId, "An unexpected error has occurred: " + e.getMessage());
    }
}
